// content.js - CapyNews content script for extracting article content
let isInitialized = false;

function initializeContentScript() {
  if (isInitialized) return;
  isInitialized = true;
  
  console.log('[CapyNews] Content script loaded for:', window.location.href);

  // Message listener
  chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
    console.log('[CapyNews] Received message:', request);
    
    if (request.action === "ping") {
      sendResponse({ status: "alive" });
      return true;
    }
    
    if (request.action === "getSelection") {
      const selectedText = window.getSelection().toString().trim();
      console.log('[CapyNews] Selected text length:', selectedText.length);

      sendResponse({
        success: true,
        text: selectedText,
        timestamp: Date.now()
      });

      // If no text selected, pass empty string - handleSelection will use full article
      handleSelection(selectedText);
    }
    return true;
  });

  console.log('[CapyNews] Message listener registered');
}

// Initialize
initializeContentScript();
document.addEventListener('DOMContentLoaded', initializeContentScript);

// Extract full article content
function extractFullArticleContent() {
  const articleSelectors = [
    'article',
    '[role="main"] article',
    '.article-content',
    '.post-content', 
    '.entry-content',
    '.content',
    '.article-body',
    '.post-body',
    '.story-body',
    '.materia-conteudo',
    '.content-text',
    '.materia-texto',
    'main',
    '.main-content',
    '#content',
    '#main'
  ];
  
  let articleElement = null;
  
  for (const selector of articleSelectors) {
    articleElement = document.querySelector(selector);
    if (articleElement && articleElement.textContent.trim().length > 200) {
      break;
    }
  }
  
  if (!articleElement) {
    articleElement = document.body;
  }
  
  const clonedElement = articleElement.cloneNode(true);
  
  // Remove unwanted elements
  const unwantedSelectors = [
    'script', 'style', 'noscript', 'iframe', 'video', 'audio', 'canvas', 'svg', 'img',
    '.ad', '.ads', '.advertisement', 
    '.social-share', '.share-buttons',
    '.comments', '.comment-section',
    '.related-articles', '.related-posts',
    '.newsletter-signup', '.subscription',
    '.header', '.navigation', '.nav',
    '.footer', '.sidebar',
    '.popup', '.modal', '.overlay',
    'aside', 'nav', 'header', 'footer',
    '.cookie-notice', '.cookie-banner',
    '.promo', '.promotion',
    '[role="complementary"]', '[role="banner"]', '[role="navigation"]', '[aria-hidden="true"]',
    '.image-caption', '.photo-credit', '.credit', '.caption',
    '.related-links', '.read-more', '.continue-reading',
    '.newsletter', '.subscribe', '.social-media', '.share',
    '.tags', '.categories', '.author-bio', '.author-info',
    '.video-player', '.player', '.video-container', '.video-wrapper',
    '.related-videos', '.video-list', '.playlist', '.video-controls', '.video-overlay',
    '.glb-video', '.glb-player', '.video-box', '.materia-video',
    '.conteudo-relacionado', '.lista-relacionados', '.box-relacionados'
  ];
  
  unwantedSelectors.forEach(selector => {
    clonedElement.querySelectorAll(selector).forEach(el => el.remove());
  });
  
  // Remove ad-related elements
  clonedElement.querySelectorAll('[class*="ad-"], [id*="ad-"], [class*="advertisement"], [id*="advertisement"]')
    .forEach(el => el.remove());
  
  // Process nodes to extract text
  const processNode = (node) => {
    if (node.nodeType === Node.TEXT_NODE) {
      const text = node.textContent.trim();
      return text ? text + ' ' : '';
    }
    
    if (node.nodeType === Node.ELEMENT_NODE) {
      const tagName = node.tagName.toLowerCase();
      const isBlockElement = ['p', 'div', 'h1', 'h2', 'h3', 'h4', 'h5', 'h6', 'li', 'blockquote', 'article', 'section'].includes(tagName);
      const isListItem = tagName === 'li';
      const isHeading = ['h1', 'h2', 'h3', 'h4', 'h5', 'h6'].includes(tagName);
      const isParagraph = tagName === 'p';
      
      let text = '';
      for (let child of node.childNodes) {
        text += processNode(child);
      }
      text = text.trim();
      
      if (text) {
        if (isHeading) return '\n\n' + text + '\n\n';
        if (isListItem) return '• ' + text + '\n';
        if (isParagraph || isBlockElement) return text + '\n\n';
        return text + ' ';
      }
    }
    
    return '';
  };
  
  let formattedText = processNode(clonedElement);
  
  // Clean up
  formattedText = formattedText
    .replace(/[ \t]+/g, ' ')
    .replace(/\n[ \t]+/g, '\n')
    .replace(/[ \t]+\n/g, '\n')
    .replace(/\n{3,}/g, '\n\n')
    .trim();
  
  return formattedText;
}

// Handle selected text - send to backend via background script
async function handleSelection(selectedText) {
  try {
    // Check auth first
    const authCheck = await chrome.runtime.sendMessage({ action: "checkAuth" });

    if (!authCheck.authenticated) {
      showNotification('Please sign in to CapyNews (click extension icon)', 'error');
      return;
    }

    // Fetch available channels
    showNotification('Loading channels...', 'info');
    const channelsData = await chrome.runtime.sendMessage({ action: "fetchChannels" });

    if (!channelsData || !channelsData.success || !channelsData.channels || channelsData.channels.length === 0) {
      // No channels configured, proceed with default
      await analyzeWithChannel(selectedText, null);
      return;
    }

    // Show channel picker if multiple channels available
    if (channelsData.channels.length > 1) {
      const selectedChannel = await showChannelPicker(channelsData.channels);
      if (!selectedChannel) {
        // User cancelled
        return;
      }
      await analyzeWithChannel(selectedText, selectedChannel);
    } else {
      // Only one channel, use it directly
      await analyzeWithChannel(selectedText, channelsData.channels[0].id);
    }

  } catch (error) {
    console.error('[CapyNews] Error:', error);
    showNotification('Error: ' + error.message, 'error');
  }
}

// Analyze article with selected channel
async function analyzeWithChannel(selectedText, channelId) {
  try {
    showNotification('Extracting article...', 'info');

    console.log('[CapyNews] Extracting full article content...');
    const fullContent = extractFullArticleContent();
    console.log('[CapyNews] Full content length:', fullContent.length);

    // If no text was selected, use the first paragraph or headline from full content
    // This gives GPT context about what the article is about
    if (!selectedText || selectedText.length === 0) {
      // Use first 500 chars as "selected text" to give GPT a hint
      selectedText = fullContent.substring(0, 500);
      console.log('[CapyNews] No selection - using article preview as selected text');
    }

    // Small delay for visual feedback
    await new Promise(resolve => setTimeout(resolve, 500));

    showNotification('Analyzing with GPT-4o...', 'info');

    // Send to background script which will call the backend
    const response = await chrome.runtime.sendMessage({
      action: "analyzeContent",
      url: window.location.href,
      selectedText: selectedText,
      fullContent: fullContent,
      channelId: channelId
    });

    console.log('[CapyNews] Response:', response);

    if (response.success) {
      if (response.slackPosted) {
        const channelName = response.channel ? response.channel.name : 'Slack';
        showNotification(`✓ Posted to ${channelName}!`, 'success');
      } else {
        showNotification('✓ Analyzed (Channel not configured)', 'warning');
      }
    } else {
      if (response.needsAuth) {
        showNotification('Please sign in (click extension icon)', 'error');
      } else {
        showNotification('Error: ' + response.error, 'error');
      }
    }

  } catch (error) {
    console.error('[CapyNews] Error:', error);
    showNotification('Error: ' + error.message, 'error');
  }
}

// Show channel picker UI
function showChannelPicker(channels) {
  return new Promise((resolve) => {
    // Remove existing picker
    const existing = document.getElementById('capynews-channel-picker');
    if (existing) existing.remove();

    // Create overlay
    const overlay = document.createElement('div');
    overlay.id = 'capynews-channel-picker';
    overlay.style.cssText = `
      position: fixed;
      top: 0;
      left: 0;
      width: 100%;
      height: 100%;
      background: rgba(0, 0, 0, 0.7);
      z-index: 999998;
      display: flex;
      align-items: center;
      justify-content: center;
      animation: capynews-fade-in 0.2s ease;
    `;

    // Create picker dialog
    const dialog = document.createElement('div');
    dialog.style.cssText = `
      background: #000;
      border: 2px solid #00d9ff;
      border-radius: 8px;
      padding: 24px;
      max-width: 400px;
      width: 90%;
      font-family: 'Courier New', Courier, monospace;
      animation: capynews-slide-up 0.3s ease;
    `;

    const title = document.createElement('div');
    title.style.cssText = `
      font-size: 16px;
      color: #00d9ff;
      margin-bottom: 8px;
      font-weight: bold;
    `;
    title.textContent = '🦫 Select Channel';

    const subtitle = document.createElement('div');
    subtitle.style.cssText = `
      font-size: 12px;
      color: #888;
      margin-bottom: 20px;
    `;
    subtitle.textContent = 'Choose where to post this article:';

    const channelList = document.createElement('div');
    channelList.style.cssText = `
      display: flex;
      flex-direction: column;
      gap: 10px;
      margin-bottom: 16px;
    `;

    channels.forEach(channel => {
      const channelBtn = document.createElement('button');
      channelBtn.style.cssText = `
        padding: 14px;
        background: #0a0a0a;
        border: 1px solid #333;
        color: #c0c0c0;
        font-family: 'Courier New', Courier, monospace;
        font-size: 13px;
        cursor: pointer;
        text-align: left;
        transition: all 0.2s;
        border-radius: 4px;
      `;

      channelBtn.innerHTML = `
        <div style="display: flex; align-items: center; gap: 10px;">
          <span style="font-size: 20px;">${channel.icon}</span>
          <div>
            <div style="color: #00d9ff; font-weight: bold; margin-bottom: 4px;">${channel.name}</div>
            <div style="font-size: 11px; color: #666;">${channel.description}</div>
          </div>
        </div>
      `;

      channelBtn.addEventListener('mouseenter', () => {
        channelBtn.style.background = '#111';
        channelBtn.style.borderColor = '#00d9ff';
      });

      channelBtn.addEventListener('mouseleave', () => {
        channelBtn.style.background = '#0a0a0a';
        channelBtn.style.borderColor = '#333';
      });

      channelBtn.addEventListener('click', () => {
        overlay.remove();
        resolve(channel.id);
      });

      channelList.appendChild(channelBtn);
    });

    const cancelBtn = document.createElement('button');
    cancelBtn.style.cssText = `
      width: 100%;
      padding: 10px;
      background: transparent;
      border: 1px solid #666;
      color: #666;
      font-family: 'Courier New', Courier, monospace;
      font-size: 12px;
      cursor: pointer;
      border-radius: 4px;
      transition: all 0.2s;
    `;
    cancelBtn.textContent = 'Cancel';

    cancelBtn.addEventListener('mouseenter', () => {
      cancelBtn.style.borderColor = '#ff6464';
      cancelBtn.style.color = '#ff6464';
    });

    cancelBtn.addEventListener('mouseleave', () => {
      cancelBtn.style.borderColor = '#666';
      cancelBtn.style.color = '#666';
    });

    cancelBtn.addEventListener('click', () => {
      overlay.remove();
      resolve(null);
    });

    dialog.appendChild(title);
    dialog.appendChild(subtitle);
    dialog.appendChild(channelList);
    dialog.appendChild(cancelBtn);
    overlay.appendChild(dialog);

    // Add animations
    const style = document.createElement('style');
    style.textContent = `
      @keyframes capynews-fade-in {
        from { opacity: 0; }
        to { opacity: 1; }
      }
      @keyframes capynews-slide-up {
        from { transform: translateY(20px); opacity: 0; }
        to { transform: translateY(0); opacity: 1; }
      }
    `;
    document.head.appendChild(style);

    document.body.appendChild(overlay);

    // Close on overlay click
    overlay.addEventListener('click', (e) => {
      if (e.target === overlay) {
        overlay.remove();
        resolve(null);
      }
    });
  });
}

// Show notification toast
function showNotification(message, type = 'info') {
  // Remove existing notification
  const existing = document.getElementById('capynews-notification');
  if (existing) existing.remove();

  const notification = document.createElement('div');
  notification.id = 'capynews-notification';

  // Terminal-style colors matching extension theme
  const colors = {
    success: { bg: '#000', border: '#00d9ff', text: '#00d9ff', icon: '✓' },
    error: { bg: '#000', border: '#ff6464', text: '#ff6464', icon: '✕' },
    warning: { bg: '#000', border: '#ffa500', text: '#ffa500', icon: '⚠' },
    info: { bg: '#000', border: '#00d9ff', text: '#00d9ff', icon: '→' }
  };

  const color = colors[type] || colors.info;

  notification.style.cssText = `
    position: fixed;
    top: 20px;
    right: 20px;
    padding: 14px 18px;
    background: ${color.bg};
    border: 2px solid ${color.border};
    border-radius: 4px;
    color: ${color.text};
    font-family: 'Courier New', Courier, monospace;
    font-size: 13px;
    font-weight: 500;
    z-index: 999999;
    box-shadow: 0 4px 16px rgba(0, 0, 0, 0.5);
    animation: capynews-slide-in 0.3s ease;
    display: flex;
    align-items: center;
    gap: 10px;
    min-width: 280px;
  `;

  notification.innerHTML = `<span style="font-size: 16px; color: ${color.text};">${color.icon}</span> <span style="color: #ddd;">${message}</span>`;
  
  // Add animation
  const style = document.createElement('style');
  style.textContent = `
    @keyframes capynews-slide-in {
      from { transform: translateX(100%); opacity: 0; }
      to { transform: translateX(0); opacity: 1; }
    }
  `;
  document.head.appendChild(style);
  
  document.body.appendChild(notification);
  
  // Auto remove
  setTimeout(() => {
    notification.style.animation = 'capynews-slide-in 0.3s ease reverse';
    setTimeout(() => notification.remove(), 300);
  }, 4000);
}
